<?php
/**
 * Plugin Name:       Podcast Importer SecondLine Pro
 * Description:       Pro version of the Podcast Importer plugin
 * Version:           1.0.4
 * Author:            SecondLineThemes
 * Author URI:        https://secondlinethemes.com/
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       podcast-importer-secondline-pro
 * Domain Path:       /languages
 */

if ( ! defined( 'WPINC' ) )
	die;

define( 'PCI_PRO_STORE_URL_PLUGIN_FILE', __FILE__ );
define( 'PODCAST_IMPORTER_PRO_SECONDLINE', 'podcast_importer_pro_secondline' );

require_once "inc/licensing.php";

function podcast_importer_pro_transcript_content_allowed_html_tags() {
  return [
    'a'      => [ 'href' => [], 'target' => [], 'alt' => [] ],
    'br'     => [],
    'video'  => [ 'width' => [], 'height' => [] ],
    'source' => [ 'src' => [], 'type' => [] ],
    'strong' => [ 'style' => [] ],
    'sub'    => [ 'style' => [] ],
    'sup'    => [ 'style' => [] ],
    's'      => [ 'style' => [] ],
    'i'      => [ 'style' => [] ],
    'u'      => [ 'style' => [] ],
    'span'   => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'h1'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'h2'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'h3'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'ol'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'ul'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'li'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'em'     => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [] ],
    'hr'     => [],
    'p'      => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'data' => [],  ],
    'img'    => [ 'align' => [], 'class' => [], 'type' => [], 'id' => [], 'style' => [], 'src' => [], 'alt' => [], 'href' => [], 'rel' => [], 'target' => [], 'value' => [], 'name' => [], 'width' => [], 'height' => [], 'data' => [], 'title' => [] ]
  ];
}

add_action( 'plugins_loaded', function() {
  if( !defined( 'PODCAST_IMPORTER_SECONDLINE_ALIAS' ) )
    return;

  add_action( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_before_feed_item_operations', function( WP_Post $post ) {
    echo '<a class="button button-secondary" 
             data-secondline-import-success-message="' . esc_attr( __( 'Synced', 'podcast-importer-secondline-pro' ) ) . '"
             data-secondline-import-rest-api-request="' . PODCAST_IMPORTER_SECONDLINE_REST_API_PREFIX . '/v1/sync-feed/' . $post->ID . '">' .
            __( 'Sync Now', 'podcast-importer-secondline-pro' ) .
         '</a>';
  });

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_scheduler_feed_args', function( $args ) {
    $option = get_post_meta( $args[ 'post_id' ], 'secondline_pro_action_schedule_interval', true );

    if( empty( $option ) || !in_array( $option, [ 'hourly', 'weekly', 'daily', '30m', '20m', '10m' ] ) )
      return $args;

    if( $option === 'hourly' ) {
      $args[ 'interval_in_seconds' ] = HOUR_IN_SECONDS;
    } else if( $option === 'weekly' || $option === 'daily' ) {
      $time_option = get_post_meta( $args[ 'post_id' ], 'secondline_pro_action_schedule_time', true );

      if( empty( $time_option ) || strlen( $time_option ) !== 5 || strpos( $time_option, ':' ) === false )
        $time_option = '08:00';

      if( $option === 'weekly' ) {
        $day_option = get_post_meta( $args[ 'post_id' ], 'secondline_pro_action_schedule_day', true );

        if( !in_array( $day_option, [ 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday' ] ) )
          $day_option = 'monday';

        $timestamp = strtotime('next ' . $day_option . $time_option . ':00' );

        if ( $timestamp < time() )
          $timestamp += WEEK_IN_SECONDS;
      } else {
        $timestamp = strtotime('today ' . $time_option . ':00' );

        if ( $timestamp < time() )
          $timestamp += DAY_IN_SECONDS;
      }

      $args[ 'interval_in_seconds' ] = ( $option === 'weekly' ? WEEK_IN_SECONDS : DAY_IN_SECONDS );
      $args[ 'timestamp' ] = $timestamp;
    } else if( $option === '30m' ) {
      $args[ 'interval_in_seconds' ] = MINUTE_IN_SECONDS * 30;
    } else if( $option === '20m' ) {
      $args[ 'interval_in_seconds' ] = MINUTE_IN_SECONDS * 20;
    } else if( $option === '10m' ) {
      $args[ 'interval_in_seconds' ] = MINUTE_IN_SECONDS * 10;
    }

    return $args;
  });

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_feed_cron_limit', function() {
    return 9999;
  } );

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_supported_post_types', function( $response ) {
    $post_types = get_post_types([ 'public' => true, 'show_in_nav_menus' => true ] );

    foreach ( $post_types as $post_type ) {
      if( in_array( $post_type, [ 'elementor_library', 'secondline_psb_post', 'secondline_import', 'secondline_shows', 'attachment', 'product', 'page' ] ) )
        continue;

      if( in_array( $post_type, $response ) )
        continue;

      $response[] = $post_type;
    }

    return $response;
  });

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_feed_form_definitions', function( $form_definitions ) {
    $form_definitions[ 'import_allow_sync' ] = [
      'label'           => __( 'Re-sync details for already imported content.', 'podcast-importer-secondline-pro'),
      'name'            => 'import_allow_sync',
      'type'            => 'checkbox',
      'value_unchecked' => 'off',
      'value_checked'   => 'on',
      'storage'         => [
        'type'  => 'meta',
        'meta'  => 'secondline_import_allow_sync'
      ]
    ];

    $form_definitions[ 'pro_import_transcript' ] = [
      'label'           => __( 'Import transcript when included in feed.', 'podcast-importer-secondline-pro'),
      'name'            => 'pro_import_transcript',
      'type'            => 'checkbox',
      'value_unchecked' => 'off',
      'value_checked'   => 'on',
      'storage'         => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_import_transcript'
      ]
    ];

    $form_definitions[ 'pro_default_image_post_id' ] = [
      'label'   => __( "Set a global featured image to all episodes", 'podcast-importer-secondline-pro' ),
      'name'    => 'pro_default_image_post_id',
      'type'    => 'media_image_id',
      'storage' => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_default_image_post_id'
      ]
    ];

    $form_definitions[ 'pro_import_category_as_taxonomy' ] = [
      'label'   => esc_html__( "Import <category> tags to a custom taxonomy", 'podcast-importer-secondline-pro' ),
      'name'    => 'pro_import_category_as_taxonomy',
      'type'    => 'select',
      'options' => [
        ''  => __( "No", 'podcast-importer-secondline-pro' )
      ] + podcast_importer_secondline_get_taxonomy_type_select_definition( podcast_importer_secondline_supported_post_types(), true ),
      'storage' => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_import_category_as_taxonomy'
      ],
      'class'       => 'secondline-post-type-logic'
    ];

    $form_definitions[ 'pro_import_itunes_keywords' ] = [
      'label'           => __( "Import keywords from RSS as tags", 'podcast-importer-secondline-pro' ),
      'name'            => 'pro_import_itunes_keywords',
      'type'            => 'checkbox',
      'value_unchecked' => 'off',
      'value_checked'   => 'on',
      'storage'         => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_import_itunes_keywords'
      ]
    ];

    $form_definitions[ 'pro_import_audio_player_meta' ] = [
      'label'   => __( "Import audio player into a custom field", 'podcast-importer-secondline-pro' ),
      'name'    => 'pro_import_audio_player_meta',
      'type'    => 'text',
      'description'     => __( 'Type the custom field name (key) here.', 'podcast-importer-secondline' ),      
      'storage' => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_import_audio_player_meta'
      ]
    ];

    $form_definitions[ 'pro_action_schedule_interval' ] = [
      'label'           => __( "Import Interval", 'podcast-importer-secondline-pro' ),
      'name'            => 'pro_action_schedule_interval',
      'type'            => 'select',
      'options'         => [
        'hourly'  => __( "Hourly ( default )", 'podcast-importer-secondline-pro' ),
        'weekly'  => __( "Weekly", 'podcast-importer-secondline-pro' ),
        'daily'   => __( "Daily", 'podcast-importer-secondline-pro' ),
        '30m'     => __( "30 min", 'podcast-importer-secondline-pro' ),
        '20m'     => __( "20 min", 'podcast-importer-secondline-pro' ),
        '10m'     => __( "10 min", 'podcast-importer-secondline-pro' )
      ],
      'storage'         => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_action_schedule_interval'
      ],
      'conditional' => '[name="import_continuous"]:1'
    ];

    $form_definitions[ 'pro_action_schedule_day' ] = [
      'label'           => __( "Import Day", 'podcast-importer-secondline-pro' ),
      'name'            => 'pro_action_schedule_day',
      'type'            => 'select',
      'options'         => [
        'monday'    => __( "Monday", 'podcast-importer-secondline-pro' ),
        'tuesday'   => __( "Tuesday", 'podcast-importer-secondline-pro' ),
        'wednesday' => __( "Wednesday", 'podcast-importer-secondline-pro' ),
        'thursday'  => __( "Thursday", 'podcast-importer-secondline-pro' ),
        'friday'    => __( "Friday", 'podcast-importer-secondline-pro' ),
        'saturday'  => __( "Saturday", 'podcast-importer-secondline-pro' ),
        'sunday'    => __( "Sunday", 'podcast-importer-secondline-pro' )
      ],
      'storage'         => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_action_schedule_day'
      ],
      'conditional' => '[name="pro_action_schedule_interval"]:weekly'
    ];

    $form_definitions[ 'pro_action_schedule_time' ] = [
      'label'           => __( "Import Time", 'podcast-importer-secondline-pro' ),
      'name'            => 'pro_action_schedule_time',
      'type'            => 'time',
      'description'     => sprintf( __( "Important: The time refers to your server current time, which is not necessarily in your personal timezone. Current Server Time : %s", "podcast-importer-secondline-pro" ), date("(D) H:i", time() ) ),
      'storage'         => [
        'type'  => 'meta',
        'meta'  => 'secondline_pro_action_schedule_time'
      ],
      'conditional' => '[name="pro_action_schedule_interval"]:weekly,daily'
    ];

    return $form_definitions;
  });

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_importer_settings_from_meta_map', function( $settings, $meta_map ) {
    if( isset( $meta_map[ 'secondline_pro_import_audio_player_meta' ] )
        && is_string( $meta_map[ 'secondline_pro_import_audio_player_meta' ] )
        && !empty( $meta_map[ 'secondline_pro_import_audio_player_meta' ] )
    )
      $settings[ 'pro_import_audio_player_meta' ] = $meta_map[ 'secondline_pro_import_audio_player_meta' ];

    if( isset( $meta_map[ 'secondline_pro_default_image_post_id' ] ) && is_numeric( $meta_map[ 'secondline_pro_default_image_post_id' ] ) )
      $settings[ 'pro_default_image_post_id' ] = $meta_map[ 'secondline_pro_default_image_post_id' ];

    if( isset( $meta_map[ 'secondline_pro_import_category_as_taxonomy' ] ) && $meta_map[ 'secondline_pro_import_category_as_taxonomy' ] !== '' )
      $settings[ 'pro_import_category_as_taxonomy' ] = $meta_map[ 'secondline_pro_import_category_as_taxonomy' ];

    if( isset( $meta_map[ 'secondline_pro_import_itunes_keywords' ] ) && $meta_map[ 'secondline_pro_import_itunes_keywords' ] !== '' )
      $settings[ 'pro_import_itunes_keywords' ] = ( $meta_map[ 'secondline_pro_import_itunes_keywords' ] === 'on' );

    if( isset( $meta_map[ 'secondline_pro_import_transcript' ] ) && $meta_map[ 'secondline_pro_import_transcript' ] !== '' )
      $settings[ 'pro_import_transcript' ] = ( $meta_map[ 'secondline_pro_import_transcript' ] === 'on' );

    return $settings;
  }, 100, 2 );

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_feed_item_has_player_in_content', function( $response, PodcastImporterSecondLine\Helper\Importer\FeedItem $feedItem ) {
    if( isset( $feedItem->importer->additional_settings[ 'pro_import_audio_player_meta' ] )
      && !empty( $feedItem->importer->additional_settings[ 'pro_import_audio_player_meta' ] )
      && !empty( $feedItem->audio_embed_html ) )
      return false;

    return $response;
  }, 100, 2 );

  add_filter( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_feed_item_import_category_map', function( $taxonomy_map, PodcastImporterSecondLine\Helper\Importer\FeedItem $feedItem ) {
    if( isset( $feedItem->importer->additional_settings[ 'pro_import_category_as_taxonomy' ] )
        && !empty( $feedItem->importer->additional_settings[ 'pro_import_category_as_taxonomy' ] ) ) {
      if( isset( $feedItem->feed_item->category ) ) {
        foreach( $feedItem->feed_item->category as $category ) {
          $category = (string) $category;

          $term_data = term_exists( $category, $feedItem->importer->additional_settings[ 'pro_import_category_as_taxonomy' ] );

          if( !isset( $term_data[ 'term_id' ] ) )
            $term_data = wp_insert_term( $category, $feedItem->importer->additional_settings[ 'pro_import_category_as_taxonomy' ] );

          if( is_wp_error( $term_data ) )
            continue;

          if( isset( $term_data[ 'term_id' ] ) )
            $taxonomy_map[ $feedItem->importer->additional_settings[ 'pro_import_category_as_taxonomy' ] ][] = $term_data[ 'term_id' ];
        }
      }
    }

    if( isset( $feedItem->importer->additional_settings[ 'pro_import_itunes_keywords' ] ) && $feedItem->importer->additional_settings[ 'pro_import_itunes_keywords' ] ) {
      if( isset( $feedItem->feed_item_itunes->keywords ) && !empty( $feedItem->feed_item_itunes->keywords ) ) {
        foreach( $feedItem->feed_item_itunes->keywords as $tags ) {
          $tags = array_map( 'trim', explode( ",", (string) $tags ) );

          if( !isset( $taxonomy_map[ 'post_tags' ] ) )
            $taxonomy_map[ 'post_tag' ] = $tags;
          else
            $taxonomy_map[ 'post_tag' ] = array_unique( array_merge( $taxonomy_map[ 'post_tags' ], $tags ) );
        }

      }
    }

    return $taxonomy_map;
  }, 100, 2 );

  add_action( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_feed_item_imported', function( PodcastImporterSecondLine\Helper\Importer\FeedItem $feedItem ) {
    if( isset( $feedItem->importer->additional_settings[ 'pro_import_audio_player_meta' ] )
        && !empty( $feedItem->importer->additional_settings[ 'pro_import_audio_player_meta' ] )
        && !empty( $feedItem->audio_embed_html ) )
      update_post_meta( $feedItem->current_post_id, $feedItem->importer->additional_settings[ 'pro_import_audio_player_meta' ], $feedItem->audio_embed_html );

    if( isset( $feedItem->importer->additional_settings[ 'pro_default_image_post_id' ] ) && is_numeric( $feedItem->importer->additional_settings[ 'pro_default_image_post_id' ] ) )
      if( empty( get_post_thumbnail_id( $feedItem->current_post_id ) ) )
        set_post_thumbnail($feedItem->current_post_id, $feedItem->importer->additional_settings[ 'pro_default_image_post_id' ] );

    if( isset( $feedItem->importer->additional_settings[ 'pro_import_transcript' ] ) && $feedItem->importer->additional_settings[ 'pro_import_transcript' ] ) {
      // There might be a much better way to approach this, but unsure...
      $string = $feedItem->feed_item->asXML();
      $identifier = '<podcast:transcript';
      $pos = strpos( $string, $identifier );

      if( $pos === false )
        return;

      $end_pos = strpos( $string, '/>', $pos );

      if( $end_pos === false )
        return;

      $string = substr( $string, $pos, $end_pos - $pos + 2 );
      $offset = strpos( $string, 'url="' );

      $url = '';
      $type = '';

      if( !empty( $offset ) ) {
        $offset += 5;

        $url = substr( $string, $offset, strpos( $string, '"', $offset ) - $offset );
      }

      if( empty( $url ) )
        return;

      $offset = strpos( $string, 'type="' );

      if( !empty( $offset ) ) {
        $offset += 6;

        $type = substr( $string, $offset, strpos( $string, '"', $offset ) - $offset );
      }

      if( !\PodcastImporterSecondLine\Helper\Scheduler::is_action_scheduled(
        PODCAST_IMPORTER_SECONDLINE_ALIAS . '_transcript_import',
        [ $feedItem->current_post_id, $url, $type ]
      ) )
        as_enqueue_async_action(
          PODCAST_IMPORTER_SECONDLINE_ALIAS . '_transcript_import',
          [ $feedItem->current_post_id, $url, $type ],
          PODCAST_IMPORTER_SECONDLINE_ALIAS
        );
    }
  });

  add_action( PODCAST_IMPORTER_SECONDLINE_ALIAS . '_transcript_import', function( $post_id, $url, $type ) {
    $content = file_get_contents( $url );

    if( !empty( $content ) ) {
      if( $type === 'text/plain' )
        $content = sanitize_textarea_field( $content );
      else
        $content = wp_kses( $content, podcast_importer_pro_transcript_content_allowed_html_tags() );

      $content = '<div class="transcript"' . ( $type === 'text/plain' ? ' style="white-space:pre-line"' : '' ) . '>' . $content . '</div>';
    }

    $post = get_post( $post_id );

    if( empty( $post ) )
      return;

    $dom = new \DOMDocument();
    @$dom->loadHTML('<?xml encoding="utf-8" ?>' . '<html><body>' . $post->post_content . '</body></html>');
    $xpath = new DOMXPath($dom);

    foreach( $xpath->query('//div[contains(attribute::class, "transcript")]') as $e ) {
      $e->parentNode->removeChild($e);
    }

    $post_content = $dom->saveHTML($dom->documentElement);
    $post_content = str_replace( [ '<html>', '<body>', '</body>', '</html>' ], '', $post_content );

    wp_update_post( [
      'ID'            => $post->ID,
      'post_content'  => $post_content . "\n" . $content
    ] );
  }, 100, 3 );
}, 9999 );